#!/usr/bin/env python3
"""
Data processor for Google Elo format.

This module handles loading and processing data in the Google Elo format,
providing compatibility with the existing Bayesian Elo framework.
"""

import pandas as pd
import numpy as np
from typing import Dict, List, Tuple, Optional
from pathlib import Path
import tempfile
import os

class GoogleEloProcessor:
    """Main processor for running Google Elo analysis."""
    
    def __init__(self, df: pd.DataFrame, valid_raters: List[str]):
        """Initialize the processor.
        
        Args:
            df: DataFrame with comparison data
            valid_raters: List of valid rater IDs
        """
        self.df = df
        self.valid_raters = valid_raters
    
    def process(self, df: pd.DataFrame = None, valid_raters: List[str] = None, uncertainty: bool = False) -> Tuple[pd.DataFrame, float]:
        """Process the data using Google Elo.
        
        Args:
            df: DataFrame with comparison data (optional, uses self.df if not provided)
            valid_raters: List of valid rater IDs (optional, uses self.valid_raters if not provided)
            model: Model name (should be 'google_elo')
            
        Returns:
            DataFrame with ELO scores and confidence intervals
        """
        if df is None:
            df = self.df
        if valid_raters is None:
            valid_raters = self.valid_raters
        
        if df.empty:
            print("No valid data found after filtering by raters")
            return pd.DataFrame(), 0
        
        try:
            # Run Google Elo analysis
            from google_elo_wrapper import GoogleEloWrapper
            wrapper = GoogleEloWrapper()
            results, time = wrapper.run_elo_analysis(df)
            
            # For Google ELO, we don't have built-in confidence intervals
            # but we can add estimated confidence intervals if uncertainty is requested
            if uncertainty and not results.empty:
                
                results['Lower CI (99%)'] = results['Lower CI (99%)'] 
                results['Upper CI (99%)'] = results['Upper CI (99%)']
            
            return results, time
            
        except Exception as e:
            print(f"Error running Google Elo analysis: {e}")
            return pd.DataFrame(), 0
